module ABEE

"""
TwoModeQBMNEGF.jl

NEGF/Keldysh steady-state covariance
for two coupled harmonic modes in phase
space R=(x1,p1,x2,p2).

- Inter-mode couplings supported: :RWA (beam-splitter), :xx, :pp, :xxpp, :none
- Bath couplings supported per mode: :x or :p (each mode has its own bath)
- Ohmic baths in the strict (Markov) limit:
    Πᴿ(ω) = -i λ ω
    Πᴷ(ω) = (Πᴿ - Πᴬ) coth(ω/2T) = -2i λ ω coth(ω/2T)
  The Caldeira–Leggett counterterm is assumed to have removed
  the static (UV-divergent) frequency shift,
  so ωⱼ in the Hamiltonian are the physical frequencies.

Fourier convention used throughout (implementation):
  F(ω) = ∫ dτ e^{-i ω τ} F(τ),  τ=t-t'.

Units:
- Internal Green's function calculations use physical coordinates (x,p) with [x,p]=i.
- Public covariance outputs are in shot-noise units (SNU): R_SNU = S * R
  with S = diag(√(2 m1 ω1), √(2/(m1 ω1)), √(2 m2 ω2), √(2/(m2 ω2))).
  In SNU, [R_SNU, R_SNU^T] = 2 i J and vacuum variances are 1.
"""

using LinearAlgebra
using QuadGK

export SystemParams, BathParams, IntegrationParams,
       symplecticJ, Kmatrix,
       SigmaR, SigmaK,
       invGR, GR, GA,
       GK, Gless, Ggreater,
       steady_state_covariance,
       symplectic_eigenvalues, partial_transpose,
       log_negativity,
       mode_stds, squeezing_dB,
       physicality_check, is_physical,
       duan_value, duan_condition


# ----------------------------
# Parameter structs
# ----------------------------

Base.@kwdef struct SystemParams
    m1::Float64 = 1.0 # Mass of the first particle.
    m2::Float64 = 1.0 # Mass of the second particle.
    ω1::Float64 = 1.0 # Frequency of the first particle.
    ω2::Float64 = 1.0 # Frequency of the second particle.
    coupling::Symbol = :RWA   # :RWA, :xx, :pp, :xxpp, :none
    # For :RWA use g
    g::Float64 = 0.0 # H ~ g ( a₁⁺a₂ + a₂⁺a₁ )
    # For :xx, :pp, :xxpp use kxx/kpp directly
    kxx::Float64 = 0.0
    kpp::Float64 = 0.0
    # Optional: override full K matrix directly (4x4).
    # If provided, coupling fields ignored.
    Kcustom::Union{Nothing, Matrix{Float64}} = nothing
end

Base.@kwdef struct BathParams
    γ::Float64 = 0.0
    T::Float64 = 0.0            # temperature (kB=1)
    coupling::Symbol = :x       # :x or :p  (couples bath to x_j or p_j)
    # Optional: directly specify λ in Πᴿ = -i λ ω, Πᴷ = -2i λ ω coth(ω/2T)
    λ::Union{Nothing, Float64} = nothing
    # Optional: Drude cutoff ωD for Ohmic spectral density.
    # If provided, uses Drude–Ohmic self-energies.
    omega_D::Union{Nothing, Float64} = nothing
    # How to interpret γ when λ is not provided:
    # - :snu      => γ is defined in SNU (dimensionless quadratures, default)
    # - :snu      => γ is defined in SNU (dimensionless quadratures)
    gamma_convention::Symbol = :snu
end

# Backward-compatible constructor for positional 4-arg calls.
BathParams(γ::Float64, T::Float64, coupling::Symbol, λ::Union{Nothing, Float64}) =
    BathParams(γ=γ, T=T, coupling=coupling, λ=λ)

BathParams(γ::Float64, T::Float64, coupling::Symbol, λ::Union{Nothing, Float64}, omega_D::Union{Nothing, Float64}) =
    BathParams(γ=γ, T=T, coupling=coupling, λ=λ, omega_D=omega_D)

Base.@kwdef struct IntegrationParams
    ωmax::Float64 = 50.0
    Nω::Int = 20001             # use an odd number so ω=0 is included
    eps::Float64 = 1e-10        # small regularization added to diagonal of invGR
end

# ----------------------------
# Helpers
# ----------------------------

"""
Symplectic form J for R=(x1,p1,x2,p2):
[J] = [[0,1,0,0],[-1,0,0,0],[0,0,0,1],[0,0,-1,0]]
"""
symplecticJ() = Float64[
    0  1  0  0;
   -1  0  0  0;
    0  0  0  1;
    0  0 -1  0
]

# index helper: mode=1,2 ; coord=:x or :p
@inline function idx(mode::Int, coord::Symbol)
    @assert mode == 1 || mode == 2
    if coord == :x
        return 1 + 2 * (mode - 1)
    elseif coord == :p
        return 2 + 2 * (mode - 1)
    else
        error("coord must be :x or :p")
    end
end

"""
Shot-noise unit (SNU) scaling matrix S for R=(x1,p1,x2,p2),
such that R_SNU = S * R and [R_SNU, R_SNU^T] = 2 i J.
"""
function snu_scale(sys::SystemParams)
    return Diagonal(Float64[
        sqrt(2 * sys.m1 * sys.ω1),
        sqrt(2 / (sys.m1 * sys.ω1)),
        sqrt(2 * sys.m2 * sys.ω2),
        sqrt(2 / (sys.m2 * sys.ω2))
    ])
end

"""
Stable computation of ω * coth(ω/2T).
- For T>0 and small |ω|, uses the limit ω*coth(ω/2T) -> 2T.
- For T==0, uses the limit ω*coth(βω/2) -> |ω|.
"""
function omega_coth(ω::Float64, T::Float64)
    if T <= 0
        return abs(ω)
    end
    x = ω / (2*T)
    ax = abs(x)
    if ax < 1e-6
        # series: coth(x) ~ 1/x + x/3 - ...
        # ω*coth(ω/2T) = 2T * (1 + x^2/3 + ...)
        return 2T * (1 + x^2/3)
    else
        # use tanh to avoid cosh/sinh overflow at large |x|
        return ω / tanh(x)
    end
end

"""
Effective λ used in Π^R = -i λ ω (strict Ohmic, counterterm removes static shift).

Defaults:
- If bath.λ is provided, use it.
- If bath.gamma_convention == :physical (legacy):
  - coupling=:x -> λ = m*γ/2
  - coupling=:p -> λ = γ/(2m)
- If bath.gamma_convention == :snu (dimensionless quadratures):
  - coupling=:x -> λ = m*ω0*γ
  - coupling=:p -> λ = γ/(m*ω0)

The :snu convention makes γ correspond to damping in shot-noise units, so
it is invariant to the chosen ω0 scaling of (x,p).
"""
function lambda_eff(bath::BathParams, m::Float64, ω0::Float64)
    if bath.λ !== nothing
        return bath.λ::Float64
    end
    if bath.gamma_convention == :physical
        if bath.coupling == :x
            # Match ladder-operator convention: Σ^R_xx = -i (γ/2) ω for m=1.
            return m * bath.γ / 2
        elseif bath.coupling == :p
            return bath.γ / (2 * m)
        else
            error("bath.coupling must be :x or :p")
        end
    elseif bath.gamma_convention == :snu
        if ω0 <= 0
            error("omega0 must be positive for :snu gamma convention")
        end
        if bath.coupling == :x
            return m * ω0 * bath.γ
        elseif bath.coupling == :p
            return bath.γ / (m * ω0)
        else
            error("bath.coupling must be :x or :p")
        end
    else
        error("gamma_convention must be :physical or :snu")
    end
end

"""
Drude factor f(ω) = ωD / (ωD - i ω).
If omega_D is not provided (or non-positive), returns 1 (strict Ohmic).
"""
function drude_factor(ω::Float64, bath::BathParams)
    ωD = bath.omega_D
    if ωD === nothing || !isfinite(ωD) || ωD <= 0
        return 1.0 + 0.0im
    end
    return ωD / (ωD - 1im * ω)
end

# ----------------------------
# System stiffness matrix K
# ----------------------------

"""
Construct 4x4 K matrix such that H = 1/2 R' K R, with R=(x1,p1,x2,p2).

Coupling choices:
- :RWA  -> κx x1x2 + κp p1p2 with κx=g*sqrt(m1ω1 m2ω2), κp=g/sqrt(m1ω1 m2ω2)
- :xx   -> kxx x1x2
- :pp   -> kpp p1p2
- :xxpp -> kxx x1x2 + kpp p1p2
- :none -> no inter-mode coupling
"""
function Kmatrix(sys::SystemParams)
    if sys.Kcustom !== nothing
        K = sys.Kcustom
        @assert size(K) == (4,4)
        return Matrix{Float64}(K)
    end

    m1,m2,ω1,ω2 = sys.m1, sys.m2, sys.ω1, sys.ω2

    κx = 0.0
    κp = 0.0
    if sys.coupling == :RWA
        s = sqrt(m1*ω1*m2*ω2)
        κx = sys.g * s
        κp = sys.g / s
    elseif sys.coupling == :xx
        κx = sys.kxx
    elseif sys.coupling == :pp
        κp = sys.kpp
    elseif sys.coupling == :xxpp
        κx = sys.kxx
        κp = sys.kpp
    elseif sys.coupling == :none
        κx = 0.0; κp = 0.0
    else
        error("Unknown sys.coupling = $(sys.coupling). Use :RWA, :xx, :pp, :xxpp, :none.")
    end

    return Float64[
        m1*ω1^2   0.0      κx       0.0;
        0.0       1/m1     0.0      κp;
        κx        0.0      m2*ω2^2  0.0;
        0.0       κp       0.0      1/m2
    ]
end

# ----------------------------
# Self-energies Σ^R and Σ^K (4x4)
# ----------------------------
"""
Retarded self-energy matrix Σ^R(ω) for two independent Ohmic baths.

Strict Ohmic with counterterm:
  Π^R(ω) = -i λ ω

If bath.omega_D is provided, uses Drude–Ohmic:
  Π^R(ω) = -i λ ω * (ωD / (ωD - i ω))
which includes the causal real part and regularizes high frequencies.
Only acts on the coordinate coupled to the bath (:x or :p) for each mode.
"""
function SigmaR(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams)
    Σ = zeros(ComplexF64, 4, 4)

    λ1 = lambda_eff(bath1, sys.m1, sys.ω1)
    λ2 = lambda_eff(bath2, sys.m2, sys.ω2)
    f1 = drude_factor(ω, bath1)
    f2 = drude_factor(ω, bath2)

    i1 = idx(1, bath1.coupling)
    i2 = idx(2, bath2.coupling)

    Σ[i1,i1] = -1im * λ1 * ω * f1
    Σ[i2,i2] = -1im * λ2 * ω * f2
    return Σ
end

"""
Keldysh self-energy matrix Σ^K(ω) for two independent thermal baths.

For each bath j:
  Π^K(ω) = (Π^R-Π^A)coth(ω/2Tj) = 2 i Im Π^R(ω) coth(ω/2Tj)

With Drude–Ohmic cutoff, Im Π^R(ω) acquires the Drude factor.
"""
function SigmaK(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams)
    Σ = zeros(ComplexF64, 4, 4)

    λ1 = lambda_eff(bath1, sys.m1, sys.ω1)
    λ2 = lambda_eff(bath2, sys.m2, sys.ω2)
    f1 = real(drude_factor(ω, bath1))
    f2 = real(drude_factor(ω, bath2))

    i1 = idx(1, bath1.coupling)
    i2 = idx(2, bath2.coupling)

    Σ[i1,i1] = -2im * λ1 * f1 * omega_coth(ω, bath1.T)
    Σ[i2,i2] = -2im * λ2 * f2 * omega_coth(ω, bath2.T)
    return Σ
end

# ----------------------------
# Green's functions
# ----------------------------

"""
Inverse retarded Green's function (4x4):
  [Gᴿ]⁻¹(ω) = -i ω J - K - Σᴿ(ω) + i * eps * I

Fourier convention: exp[-i ω (t-t')].
"""
function invGR(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10)
    J = symplecticJ()
    K = Kmatrix(sys)
    ΣR = SigmaR(ω, sys, bath1, bath2)
    M = -1im * ω .* ComplexF64.(J) .- ComplexF64.(K) .- ΣR
    # tiny regularization for numerical stability
    M .+= 1im*eps .* I(4)
    return M
end

"""
Retarded Green's function G^R(ω) = inv(invGR(ω)).
"""
GR(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10) =
    inv(invGR(ω, sys, bath1, bath2; eps=eps))

"""
Advanced Green's function: G^A(ω) = (G^R(ω))†
"""
GA(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10) =
    adjoint(GR(ω, sys, bath1, bath2; eps=eps))

"""
Keldysh Green's function:
  G^K = G^R Σ^K G^A
"""
function GK(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10)
    GRω = GR(ω, sys, bath1, bath2; eps=eps)
    GAω = adjoint(GRω)
    ΣK = SigmaK(ω, sys, bath1, bath2)
    return GRω * ΣK * GAω
end

"""
Lesser and greater GFs from (GR, GA, GK):
For bosons (with our sign conventions):
  GK = G> + G<
  GR - GA = G> - G<
=> G< = 1/2 (GK - (GR - GA))
   G> = 1/2 (GK + (GR - GA))
"""
function Gless(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10)
    GRω = GR(ω, sys, bath1, bath2; eps=eps)
    GAω = adjoint(GRω)
    GKω = GRω * SigmaK(ω, sys, bath1, bath2) * GAω
    return 0.5 .* (GKω .- (GRω .- GAω))
end

function Ggreater(ω::Float64, sys::SystemParams, bath1::BathParams, bath2::BathParams; eps::Float64=1e-10)
    GRω = GR(ω, sys, bath1, bath2; eps=eps)
    GAω = adjoint(GRω)
    GKω = GRω * SigmaK(ω, sys, bath1, bath2) * GAω
    return 0.5 .* (GKω .+ (GRω .- GAω))
end

# ----------------------------
# Steady-state covariance
# ----------------------------

"""
Compute steady-state covariance matrix V (4x4) for R=(x1,p1,x2,p2):
  GK(t,t) = ∫ dω/(2π) GK(ω)
and with GK(t,t) = -i <{R,R^T}> = -2i V, we have
  V = (i/2) ∫ dω/(2π) GK(ω).

Numerical integration: trapezoidal rule over ω ∈ [-ωmax, ωmax] with Nω points.

Returns V in shot-noise units (SNU), where vacuum variances are 1 and
[R_SNU, R_SNU^T] = 2 i J.

If `check_physicality=true`, performs a Robertson–Schrödinger check on V (SNU)
and either errors or warns based on `throw_on_unphysical`.
"""
function steady_state_covariance(sys::SystemParams,
                                 bath1::BathParams,
                                 bath2::BathParams,
                                 integ::IntegrationParams;
                                 method::Symbol=:trapz,
                                 quadgk_rtol::Float64=1e-8,
                                 quadgk_atol::Float64=1e-10,
                                 quadgk_points::Vector{Float64}=Float64[],
                                 quadgk_maxevals::Int=0,
                                 quadgk_order::Int=7,
                                 check_physicality::Bool=true,
                                 physicality_tol::Float64=1e-10,
                                 throw_on_unphysical::Bool=true)
    if method == :quadgk
        p = isempty(quadgk_points) ? Float64[-sys.ω2, -sys.ω1, 0.0, sys.ω1, sys.ω2] : quadgk_points
        p = sort(unique(p))
        bounds = vcat(-Inf, p, Inf)
        integrand = ω -> vec(GK(ω, sys, bath1, bath2; eps=integ.eps))
        acc = zeros(ComplexF64, 16)
        for k in 1:length(bounds)-1
            a, b = bounds[k], bounds[k+1]
            vk, _ = quadgk(integrand, a, b;
                           rtol=quadgk_rtol,
                           atol=quadgk_atol/length(bounds),
                           maxevals=quadgk_maxevals,
                           order=quadgk_order)
            acc .+= vk
        end
        GK_steady = reshape(acc, 4, 4) / (2π)
        Vc = (1im/2) * GK_steady
        V = real.( (Vc .+ transpose(Vc)) ./ 2 )
        S = snu_scale(sys)
        V_snu = S * V * S
        if check_physicality
            phys = physicality_check(V_snu; tol=physicality_tol)
            if !phys.ok
                msg = "Unphysical covariance: min_eig=$(phys.min_eig)"
                throw_on_unphysical ? error(msg) : @warn msg
            end
        end
        return Matrix{Float64}(V_snu)
    elseif method != :trapz
        error("Unknown integration method: $(method). Use :trapz or :quadgk.")
    end
    ωmax, Nω, eps = integ.ωmax, integ.Nω, integ.eps
    @assert Nω >= 3
    @assert isodd(Nω) "Nω must be odd so ω=0 is included."
    ωs = range(-ωmax, ωmax; length=Nω)
    dω = step(ωs)

    acc = zeros(ComplexF64, 4, 4)
    for (k, ω) in enumerate(ωs)
        w = (k == 1 || k == Nω) ? 0.5 : 1.0
        acc .+= w .* GK(ω, sys, bath1, bath2; eps=eps)
    end

    Vc = (1im/2) * (dω/(2π)) * acc

    # Enforce real symmetric covariance (numerical noise may leave tiny imag/antisymm parts)
    V = real.( (Vc .+ transpose(Vc)) ./ 2 )
    # Convert to SNU for all public outputs/diagnostics.
    S = snu_scale(sys)
    V_snu = S * V * S
    if check_physicality
        phys = physicality_check(V_snu; tol=physicality_tol)
        if !phys.ok
            msg = "Unphysical covariance: min_eig=$(phys.min_eig)"
            throw_on_unphysical ? error(msg) : @warn msg
        end
    end
    return Matrix{Float64}(V_snu)
end

# ----------------------------
# Symplectic utilities & entanglement / squeezing diagnostics
# ----------------------------

# General symplectic form for n modes; keep symplecticJ() = symplecticJ(2)
function symplecticJ(n::Int)
    @assert n >= 1
    J2 = Float64[0 1; -1 0]
    J = zeros(Float64, 2n, 2n)
    for k in 1:n
        J[2k-1:2k, 2k-1:2k] .= J2
    end
    return J
end
symplecticJ() = symplecticJ(2)

"""
Return symplectic eigenvalues ν_k (k=1..n) of a 2n×2n covariance matrix V (SNU),
defined as the positive eigenvalues of |eig(i J V)|.

Returns a length-n Vector{Float64} sorted ascending.
"""
function symplectic_eigenvalues(V::AbstractMatrix{<:Real}; tol::Float64=1e-9)
    N = size(V, 1)
    @assert size(V, 2) == N "V must be square."
    @assert iseven(N) "V dimension must be even (2n)."
    n = N ÷ 2

    J = ComplexF64.(symplecticJ(n))
    M = 1im * J * ComplexF64.(V)
    ev = eigvals(M)
    vals = sort(abs.(ev))

    # Eigenvalues come in (approximately) degenerate pairs; collect n unique values robustly
    ν = Float64[]
    for v in vals
        if isempty(ν) || abs(v - ν[end]) > tol
            push!(ν, v)
            length(ν) == n && break
        end
    end

    # Fallback if degeneracy collection failed due to numerical issues:
    if length(ν) < n
        ν = vals[1:n]
    end
    return ν
end

"""
Partial transpose of a 4×4 covariance matrix for two modes in ordering (x1,p1,x2,p2).
Implements PT by flipping the momentum of the selected mode: p -> -p.

mode = 1 flips p1, mode = 2 flips p2.
"""
function partial_transpose(V::AbstractMatrix{<:Real}; mode::Int=2)
    @assert size(V) == (4,4) "partial_transpose expects 4×4 V for two modes."
    @assert mode == 1 || mode == 2
    Λ = Matrix{Float64}(I, 4, 4)
    Λ[idx(mode, :p), idx(mode, :p)] = -1.0
    return Λ * Matrix{Float64}(V) * Λ
end

"""
Gaussian logarithmic negativity for a two-mode covariance matrix V (4×4), SNU.

ν̃_- is smallest symplectic eigenvalue of the partially transposed covariance V^(PT).
E_N = max(0, -log2(ν̃_-))  (entangled iff ν̃_- < 1)
"""
function log_negativity(V::AbstractMatrix{<:Real}; mode::Int=2)::Float64
    Vpt = partial_transpose(V; mode=mode)
    νpt = symplectic_eigenvalues(Vpt)
    νmin = νpt[1]
    EN = -log(νmin) / log(2.0)
    return max(0.0, EN)
end

"""
Return standard deviations (σx, σp) for a given mode (1 or 2) from a 4×4 covariance V
in shot-noise units (vacuum σ = 1). Ordering assumed: (x1,p1,x2,p2).
"""
function mode_stds(V::AbstractMatrix{<:Real}; mode::Int=1)
    @assert size(V) == (4,4) "mode_stds expects 4×4 V."
    @assert mode == 1 || mode == 2
    ix = idx(mode, :x)
    ip = idx(mode, :p)
    σx = sqrt(max(V[ix,ix], 0.0))
    σp = sqrt(max(V[ip,ip], 0.0))
    return (σx=σx, σp=σp)
end

"""
Quadrature squeezing in dB for a mode, referenced to the *vacuum* variances in SNU
(Var(x)_vac = Var(p)_vac = 1).

Returns (sx_dB, sp_dB) where negative means squeezed below vacuum.
"""
function squeezing_dB(V::AbstractMatrix{<:Real}, sys::SystemParams; mode::Int=1)
    @assert size(V) == (4,4) "squeezing_dB expects 4×4 V."
    @assert mode == 1 || mode == 2
    ix = idx(mode, :x)
    ip = idx(mode, :p)

    # Avoid log10 of non-positive variances from numerical noise.
    var_floor = eps(Float64)
    varx = max(V[ix,ix], var_floor)
    varp = max(V[ip,ip], var_floor)

    varx_vac = 1.0
    varp_vac = 1.0

    sx = 10.0 * log10(varx / varx_vac)
    sp = 10.0 * log10(varp / varp_vac)
    return (sx_dB=sx, sp_dB=sp)
end

"""
Physicality (Robertson–Schrödinger) check in SNU: V + i J must be positive semidefinite.

Returns (ok::Bool, min_eig::Float64), where min_eig is the smallest eigenvalue of
the Hermitian matrix H = V + i J.
"""
function physicality_check(V::AbstractMatrix{<:Real}; tol::Float64=1e-10)
    N = size(V,1)
    @assert size(V,2) == N
    @assert iseven(N)
    n = N ÷ 2
    J = ComplexF64.(symplecticJ(n))
    H = Hermitian(ComplexF64.(V) .+ (1im) .* J)
    evals = eigvals(H)
    min_eig = minimum(real.(evals))
    return (ok = min_eig >= -tol, min_eig = min_eig)
end

"""
Convenience boolean-only physicality check.
"""
is_physical(V::AbstractMatrix{<:Real}; tol::Float64=1e-10) = physicality_check(V; tol=tol).ok

# Variance of a linear combination c'R where V is covariance: Var(c'R)=c' V c
@inline function _linvar(V::AbstractMatrix{<:Real}, c::AbstractVector{<:Real})
    return dot(c, V * c)
end

"""
Compute the Duan quantity for two modes (4×4 covariance, ordering x1,p1,x2,p2), SNU.

Standard Duan form (Duan et al. PRL 84, 2722 (2000)) uses:
  u = a x1 + (1/a) x2
  v = a p1 - (1/a) p2
and separable states satisfy (SNU):
  Var(u) + Var(v) ≥ 2 (a^2 + 1/a^2)

You can choose the sign structure via `form`:
  :plusminus  -> u = a x1 + (1/a) x2, v = a p1 - (1/a) p2   (EPR-like)
  :minusplus  -> u = a x1 - (1/a) x2, v = a p1 + (1/a) p2   (equivalent by local π phase)
Returns (duan::Float64, bound::Float64).
"""
function duan_value(V::AbstractMatrix{<:Real}; a::Float64=1.0, form::Symbol=:plusminus)
    @assert size(V) == (4,4) "duan_value expects 4×4 V."
    @assert a > 0

    sx = 1.0
    sp = -1.0
    if form == :plusminus
        sx = +1.0; sp = -1.0
    elseif form == :minusplus
        sx = -1.0; sp = +1.0
    else
        error("form must be :plusminus or :minusplus")
    end

    c_u = zeros(Float64, 4)
    c_v = zeros(Float64, 4)

    c_u[idx(1,:x)] = a
    c_u[idx(2,:x)] = (sx/a)

    c_v[idx(1,:p)] = a
    c_v[idx(2,:p)] = (sp/a)

    var_u = _linvar(V, c_u)
    var_v = _linvar(V, c_v)

    duan = var_u + var_v
    bound = 2.0 * (a^2 + 1.0/a^2)
    return (duan=duan, bound=bound)
end

"""
Duan entanglement test.

Returns (entangled::Bool, duan::Float64, bound::Float64).
By Duan criterion, entangled if duan < bound (strict inequality).
"""
function duan_condition(V::AbstractMatrix{<:Real}; a::Float64=1.0, form::Symbol=:plusminus, tol::Float64=0.0)
    d = duan_value(V; a=a, form=form)
    return (entangled = d.duan < d.bound - tol, duan = d.duan, bound = d.bound)
end


end # module
